#include <iostream>
#include <vector>
#include <string>
#include <sstream>
#include <algorithm>
#include <cstdint>

// convert dot-decimal IPv4 string to uint32_t
uint32_t ip_to_int(const std::string &ip_str)
{
	uint32_t ip_int = 0;
	std::stringstream ss(ip_str);
	std::string segment;
	for (int i = 0; i < 4; ++i)
	{
		std::getline(ss, segment, '.');
		ip_int = (ip_int << 8) | std::stoul(segment);
	}
	return ip_int;
}

// Function to convert uint32_t to dot-decimal IPv4 string
std::string int_to_ip(uint32_t ip_int)
{
	std::string ip_str;

	for (int i = 0; i < 4; ++i)
	{
		ip_str = std::to_string(ip_int & 255) + (i == 0 ? "" : ".") + ip_str;
		ip_int >>= 8;
	}

	return ip_str;
}

enum Action
{
	ALLOW,
	DENY
};

struct Rule
{
	Action action;
	uint32_t ip_start;
	uint32_t ip_end;
	uint16_t port_start;
	uint16_t port_end;

	void print() const
	{
		std::cout << (action == ALLOW ? "ALLOW" : "DENY") << " "
				  << int_to_ip(ip_start) << " "
				  << int_to_ip(ip_end) << " "
				  << port_start << " "
				  << port_end << std::endl;
	}
};

// check if rule_a's IP and port ranges are completely contained within rule_b's
bool is_contained(const Rule &rule_a, const Rule &rule_b)
{
	bool ip_contained = (rule_b.ip_start <= rule_a.ip_start && rule_a.ip_end <= rule_b.ip_end);
	bool port_contained = (rule_b.port_start <= rule_a.port_start && rule_a.port_end <= rule_b.port_end);
	return ip_contained && port_contained;
}

int main()
{
	std::ios_base::sync_with_stdio(false);
	std::cin.tie(NULL);

	int N;
	std::cin >> N;

	std::vector<Rule> minimized_rules;

	for (int i = 0; i < N; ++i)
	{
		std::string action_str;
		std::string ip_start_str, ip_end_str;
		uint16_t port_start, port_end;

		std::cin >> action_str >> ip_start_str >> ip_end_str >> port_start >> port_end;

		Rule current_rule;
		current_rule.action = (action_str == "ALLOW" ? ALLOW : DENY);
		current_rule.ip_start = ip_to_int(ip_start_str);
		current_rule.ip_end = ip_to_int(ip_end_str);
		current_rule.port_start = port_start;
		current_rule.port_end = port_end;

		bool is_redundant = false;

		// check if the current rule is fully covered by any previously kept rule
		for (const Rule &kept_rule : minimized_rules)
		{
			if (is_contained(current_rule, kept_rule))
			{
				is_redundant = true;
				break; // found a covering rule, current_rule is redundant.
			}
		}

		if (!is_redundant)
		{
			minimized_rules.push_back(current_rule);
		}
	}

	// print the minimized ruleset
	for (const Rule &rule : minimized_rules)
	{
		rule.print();
	}

	return 0;
}